<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

// Check if admin
if (!isAdmin()) {
    die('Access denied');
}

$pageTitle = 'Debug eSIM Creation';

// Get order ID from URL
$orderId = $_GET['order'] ?? 0;

if (!$orderId) {
    die('No order ID provided');
}

// Get order
$order = getOrderById($orderId);

if (!$order) {
    die('Order not found');
}

// Try to create eSIM
$api = new eSIMAPI();
$result = null;
$error = null;

if (isset($_POST['create_esim'])) {
    try {
        error_log("DEBUG: Creating eSIM for package: " . $order['package_code']);
        $result = $api->createESIMOrder($order['package_code'], 1);
        
        error_log("DEBUG: API Response: " . json_encode($result));
        
        if ($result === false || $result === null) {
            $error = 'API returned NULL or FALSE. Check API credentials and network connection.';
        } elseif (isset($result['success']) && $result['success'] === true) {
            // Order created successfully, now query the eSIM details
            $orderNo = $result['obj']['orderNo'] ?? null;
            
            if ($orderNo) {
                error_log("DEBUG: Order created successfully, orderNo: $orderNo. Querying eSIM details...");
                
                // Wait a moment for the eSIM to be allocated
                sleep(2);
                
                // Query the eSIM details
                $queryResult = $api->queryESIMByOrderNo($orderNo);
                error_log("DEBUG: Query eSIM response: " . json_encode($queryResult));
                
                if ($queryResult && isset($queryResult['success']) && $queryResult['success'] === true) {
                    $esimList = $queryResult['obj']['esimList'] ?? [];
                    
                    if (!empty($esimList)) {
                        $esim = $esimList[0]; // Get first eSIM
                        
                        $esimData = [
                            'esim_id' => $esim['esimTranNo'] ?? '',
                            'iccid' => $esim['iccid'] ?? '',
                            'qr_code' => $esim['qrCodeUrl'] ?? '',
                            'activation_code' => $esim['ac'] ?? ''
                        ];
                        
                        updateOrderWithESIM($order['id'], $esimData);
                        $success = 'eSIM created successfully! ICCID: ' . $esimData['iccid'];
                        
                        // Reload order
                        $order = getOrderById($orderId);
                        
                        // Store result for display
                        $result = $queryResult;
                    } else {
                        $error = 'eSIM list is empty for orderNo: ' . $orderNo . '. Response: ' . json_encode($queryResult);
                    }
                } else {
                    $error = 'Failed to query eSIM details for orderNo: ' . $orderNo . '. Response: ' . json_encode($queryResult);
                }
            } else {
                $error = 'OrderNo not found in API response. Response: ' . json_encode($result);
            }
        } else {
            $errorCode = $result['errorCode'] ?? 'unknown';
            $errorMsg = $result['errorMsg'] ?? 'Unknown error';
            $error = 'API returned error - Code: ' . $errorCode . ', Message: ' . $errorMsg;
        }
    } catch (Exception $e) {
        $error = 'Exception: ' . $e->getMessage();
        error_log("DEBUG: Exception creating eSIM: " . $e->getMessage());
    }
}

include __DIR__ . '/includes/header.php';
?>

<style>
.debug-container {
    max-width: 1000px;
    margin: 0 auto;
    background: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 5px 25px rgba(0, 0, 0, 0.1);
}

.debug-section {
    margin: 20px 0;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
}

.debug-section h3 {
    color: #667eea;
    margin-bottom: 15px;
}

.debug-data {
    background: #2d3748;
    color: #68d391;
    padding: 15px;
    border-radius: 8px;
    font-family: monospace;
    overflow-x: auto;
    white-space: pre-wrap;
    word-wrap: break-word;
}

.alert {
    padding: 15px;
    border-radius: 8px;
    margin: 15px 0;
}

.alert-success {
    background: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
}

.alert-error {
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
}

.btn {
    display: inline-block;
    padding: 12px 30px;
    background: #667eea;
    color: white;
    text-decoration: none;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    font-size: 16px;
}

.btn:hover {
    background: #5568d3;
}

.btn-danger {
    background: #dc3545;
}

.btn-danger:hover {
    background: #c82333;
}

table {
    width: 100%;
    border-collapse: collapse;
}

table td {
    padding: 10px;
    border-bottom: 1px solid #ddd;
}

table td:first-child {
    font-weight: bold;
    color: #666;
    width: 200px;
}
</style>

<div class="container">
    <div class="debug-container">
        <h1>🔧 Debug eSIM Creation</h1>
        
        <?php if (isset($success)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <?php echo $success; ?>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-error">
            <i class="fas fa-exclamation-circle"></i>
            <?php echo htmlspecialchars($error); ?>
        </div>
        <?php endif; ?>
        
        <div class="debug-section">
            <h3>📋 Order Information</h3>
            <table>
                <tr>
                    <td>Order ID:</td>
                    <td>#<?php echo $order['id']; ?></td>
                </tr>
                <tr>
                    <td>Status:</td>
                    <td><?php echo $order['status']; ?></td>
                </tr>
                <tr>
                    <td>Package Code:</td>
                    <td><?php echo htmlspecialchars($order['package_code']); ?></td>
                </tr>
                <tr>
                    <td>Package Name:</td>
                    <td><?php echo htmlspecialchars($order['package_name']); ?></td>
                </tr>
                <tr>
                    <td>Country:</td>
                    <td><?php echo htmlspecialchars($order['country_name']); ?> (<?php echo $order['country_code']; ?>)</td>
                </tr>
                <tr>
                    <td>Has eSIM Data:</td>
                    <td><?php echo !empty($order['iccid']) ? '✅ Yes' : '❌ No'; ?></td>
                </tr>
            </table>
        </div>
        
        <?php if (!empty($order['iccid'])): ?>
        <div class="debug-section">
            <h3>✅ eSIM Data</h3>
            <table>
                <tr>
                    <td>eSIM ID:</td>
                    <td><?php echo htmlspecialchars($order['esim_id']); ?></td>
                </tr>
                <tr>
                    <td>ICCID:</td>
                    <td><?php echo htmlspecialchars($order['iccid']); ?></td>
                </tr>
                <tr>
                    <td>Activation Code:</td>
                    <td><?php echo htmlspecialchars($order['activation_code']); ?></td>
                </tr>
                <tr>
                    <td>QR Code:</td>
                    <td>
                        <?php if ($order['qr_code']): ?>
                            <img src="<?php echo htmlspecialchars($order['qr_code']); ?>" style="max-width: 200px;">
                        <?php else: ?>
                            N/A
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>
        <?php endif; ?>
        
        <?php if ($result): ?>
        <div class="debug-section">
            <h3>📡 API Response</h3>
            <div class="debug-data"><?php echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE); ?></div>
        </div>
        <?php endif; ?>
        
        <div class="debug-section">
            <h3>🔄 Actions</h3>
            <form method="POST">
                <button type="submit" name="create_esim" class="btn">
                    <i class="fas fa-plus"></i>
                    <?php echo !empty($order['iccid']) ? 'Re-create eSIM' : 'Create eSIM'; ?>
                </button>
            </form>
            <br>
            <a href="/admin/orders.php" class="btn">
                <i class="fas fa-arrow-left"></i>
                Back to Orders
            </a>
        </div>
        
        <div class="debug-section">
            <h3>📝 PHP Error Log (Last 50 lines)</h3>
            <div class="debug-data">
                <?php
                $logFile = ini_get('error_log');
                if (file_exists($logFile)) {
                    $lines = file($logFile);
                    $lastLines = array_slice($lines, -50);
                    echo htmlspecialchars(implode('', $lastLines));
                } else {
                    echo "Log file not found: $logFile\n\n";
                    echo "Try checking:\n";
                    echo "- /home/matinacard/logs/error_log\n";
                    echo "- /var/log/apache2/error.log\n";
                    echo "- /var/log/nginx/error.log\n";
                }
                ?>
            </div>
        </div>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>

